% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layers.R
\name{sass_layer}
\alias{sass_layer}
\alias{sass_bundle}
\alias{sass_bundle_remove}
\alias{is_sass_bundle}
\title{Bundling Sass layers}
\usage{
sass_layer(
  defaults = NULL,
  declarations = NULL,
  rules = NULL,
  html_deps = NULL,
  file_attachments = character(0),
  tags = NULL
)

sass_bundle(...)

sass_bundle_remove(bundle, name)

is_sass_bundle(x)
}
\arguments{
\item{defaults}{A suitable \code{\link[=as_sass]{as_sass()}} \code{input}. Intended for declaring
variables with \code{!default}. When layers are combined, defaults are merged in
reverse order; that is, \code{sass_bundle(layer1, layer2)} will include
\code{layer2$defaults} before \code{layer1$defaults}.}

\item{declarations}{A suitable \code{\link[=as_sass]{as_sass()}} \code{input}.  Intended for
function and mixin declarations, and variable declarations without
\code{!default}; not intended for actual CSS rules. These will be merged in
forward order; that is, \code{sass_bundle(layer1, layer2)} will include
\code{layer1$declarations} before \code{layer2$declarations}.}

\item{rules}{A suitable \code{\link[=as_sass]{as_sass()}} \code{input}. Intended for actual CSS
rules. These will be merged in forward order; that is,
\code{sass_bundle(layer1, layer2)} will include \code{layer1$rules} before
\code{layer2$rules}.}

\item{html_deps}{An HTML dependency (or a list of them).}

\item{file_attachments}{A named character vector, representing file assets
that are referenced (using relative paths) from the sass in this layer. The
vector names should be a relative path, and the corresponding vector values
should be absolute paths to files or directories that exist; at render
time, each value will be copied to the relative path indicated by its name.
(For directories, the \emph{contents} of the source directory will be copied
into the destination directory; the directory itself will not be copied.)
You can also omit the name, in which case that file or directory will be
copied directly into the output directory.}

\item{tags}{Deprecated. Preserve meta information using a key in \code{sass_bundle(KEY = val)}.
preserve simple metadata as layers are merged.}

\item{...}{A collection of \code{sass_layer()}s and/or objects that \code{\link[=as_sass]{as_sass()}}
understands. Arguments should be provided in reverse priority order:
defaults, declarations, and rules in later layers will take precedence over
those of previous layers. Non-layer values will be converted to layers by
calling \code{sass_layer(rules = ...)}.}

\item{bundle}{Output value from \code{sass_layer()} or \code{sass_bundle()}}

\item{name}{If a Sass layer name is contained in \code{name}, the matching Sass layer will be removed from the \code{bundle}}

\item{x}{object to inspect}
}
\description{
Sass layers are a way to group a set of related Sass variable definitions,
function/mixin declarations, and CSS rules into a single object. Use
\code{sass_layer()} to create these objects, and \code{sass_bundle()} to combine
two or more layers or bundles objects into a Sass bundle; this ability to be merged is
the main benefit of using Sass layers versus lower-level forms of sass input.
At a later time, Sass layers may be removed from Sass bundles
by referencing the same name that was used when creating the Sass bundle.
}
\section{Functions}{
\itemize{
\item \code{sass_layer}: Compose the parts of a single Sass layer. Object returned is a \code{sass_bundle()} with a single Sass layer

\item \code{sass_bundle}: Collect \code{sass_bundle()} and/or \code{sass_layer()} objects. Unnamed Sass bundles will be concatenated together, preserving their internal name structures. Named Sass bundles will be condensed into a single Sass layer for easier removal from the returned Sass bundle.

\item \code{sass_bundle_remove}: Remove a whole \code{sass_layer()} from a \code{sass_bundle()} object.

\item \code{is_sass_bundle}: Check if \code{x} is a Sass bundle object
}}

\examples{
blue <- list(color = "blue !default")
red <- list(color = "red !default")
green <- list(color = "green !default")

# a sass_layer() by itself is not very useful, it just defines some
# SASS to place before (defaults) and after (declarations, rules)
core <- sass_layer(defaults = blue, rules = "body { color: $color; }")
core
sass(core)

# However, by stacking sass_layer()s, we have ability to place
# SASS both before and after some other sass (e.g., core)
# Here we place a red default _before_ the blue default and export the
# color SASS variable as a CSS variable _after_ the core
red_layer <- sass_layer(red, rules = ":root{ --color: #{$color}; }")
sass(sass_bundle(core, red_layer))
sass(sass_bundle(core, red_layer, sass_layer(green)))

# Example of merging layers and removing a layer
# Remember to name the layers that are removable
core_layers <- sass_bundle(core, red = red_layer, green = sass_layer(green))
core_layers # pretty printed for console
core_slim <- sass_bundle_remove(core_layers, "red")
sass(core_slim)


# File attachment example: Create a checkboard pattern .png, then
# use it from a sass layer

tmp_png <- tempfile(fileext = ".png")
grDevices::png(filename = tmp_png, width = 20, height = 20,
  bg = "transparent", antialias = "none")
par(mar = rep_len(0,4), xaxs = "i", yaxs = "i")
plot.new()
rect(c(0,0.5), c(0,0.5), c(0.5,1), c(0.5,1), col = "#00000044", border=NA)
dev.off()

layer <- sass_layer(
  rules = ".bg-check { background-image: url(images/demo_checkboard_bg.png) }",
  file_attachments = c("images/demo_checkboard_bg.png" = tmp_png)
)

output_path <- tempfile(fileext = ".css")
sass(layer, output = output_path, write_attachments = TRUE)
}
